<?php

/**
 * Classe Archiviste
 * 
 * Permet de manipuler les données des fichiers ainsi que de faire des recherches
 * en utilisant des objets issue de la classe Archivable.
 * 
 * /!\ Pensez a modifier l'attribut $cheminBase en fonction de l'architecture 
 * de votre programme
 * 
 * @version : 2.2
 * @date : 2021
 * @author : Selenith (https://selenith.madyweb.net)
 * @license : GPLv3 (https://www.gnu.org/licenses/gpl-3.0.html)
 * ------------------------
 * fonctions :
 * 	-archive($archivable) :
 * 		Enregistre l'objet dans la table correspondante.
 * 		Le champ "id" est auto-incrementé.
 * 		Renvoi l'id de l'objet enregistré.
 * 	
 * 	-restore(archivable) :
 * 		Renvoi la liste des objet dont les attributs correspondent au attributs
 * 		non null de l'objet passé en parametre		
 * 		Renvoi Archivable[] la liste des objet contenu dans la table en fonction des criteres de recherche.
 * 
 * 	-restore_first(archivable) :
 * 		Renvoi la liste des objet dont les attributs correspondent au attributs
 * 		non null de l'objet passé en parametre		
 * 		Renvoi Archivable : le premier des objets contenu dans la table en fonction des criteres de recherche.
 * 
 * 	-search($archivable) :
 * 		Renvoi la liste des objet dont les attributs ressemblent au attributs
 * 		non null de l'objet passé en parametre		
 * 		Renvoi Archivable[] la liste des objet contenu dans la table en fonction des criteres de recherche.
 * 
 * 	-update($archivableAncien, $archivableNouveau) :
 * 		Modifie les attributs des objets archivables $archivableAncien par les attributs de l'objet $archivableNouveau
 * 		La recherche des objets et basé sur les attributs non vides et non null de l'objet $archivableAncien.
 * 		Seul les attribut non nulls du nouvel objet ecraseron les attributs de l'ancien		
 * 		Renvoi true si une modification à été effectuée.
 * 		
 * 	-delete($archivable) :
 * 		Fonction supprimant du systeme de persistance les objet
 * 		correspondant a l'archivable passé en parametre
 * 		Renvoi true si une suppression à été effectuée.
 * 		
 * 	-sort($liste, $critere, $sens)
 * 		Permet de trier une liste d'objet archivables en fonction
 * 		d'un de ses attributs.
 * 		si $sens = true : ordre alphabetique
 * 		si $sens = false : ordre inverse
 * 		Retourne la liste des objets trié. 
 * 
 */
class Archivist{	
	
	//A modifier au besoin
	private $cheminBase = '';
			
	
	//initialisation de la position de la database	
	function __construct($dataPath = 'local/data/'){
		$this->cheminBase = $dataPath;
	}
		
	
	//enregistre l'objet dans la table correspondante
	public function archive($archivable){
		
		//recuperation du nom de la table
		$nomTable = $archivable->get_archivable_name();		
		$dbPath = $this->cheminBase;

		
		//creation d'un accesseur a la base de donnée
		$ribosome = new Ribosome($dbPath);
		//etablisement de la connexion et verouillage de la table
		$ribosome->connexion($nomTable);		
		$index = $archivable->get('id');
		if(isset($index)){
				
			$archivable->set('id', $index);
		}else{
			$index = $ribosome->indexMax() +1;		
			$archivable->set('id', $index);
		}
		
		$attributs = $archivable->getAttributs();	
		
		//$elements[$index] = $attributs;
				
		//$ribosome->ecrire($elements);	
		$ribosome->ecrire($attributs);	
		//le retour est l'index de l'élément créé
		return $index;
	}
	
	/**
	* Renvoi la liste des objet dont les attributs correspondent au attributs
	* non null de l'objet passé en parametre
	* @return Archivable[] la liste des objet contenu dans la table en fonction des criteres de recherche.
	*/
	public function restore($archivable){
	
		$nomTable = $archivable->get_archivable_name();
		$dbPath = $this->cheminBase;

		//creation d'un accesseur a la base de donnée
		$ribosome = new Ribosome($dbPath);
		$ribosome->connexion($nomTable);
		
		$elemsTrouve = array();		
			
		$attributs = $archivable->getAttributs();
		
		
		$elements = $ribosome->donnerTraduction($attributs);			
		
				
		$nbCles = count($elements);	
		
		$index = 0;
		
		for($i = 0; $i< $nbCles; $i++){			
			if($this->correspond($elements[$i], $attributs)){				
			
				$elemsTrouve[$index] = new Archivable($nomTable);
				$elemsTrouve[$index]->setAttributs($elements[$i]);
				$index++;					
			}
		}
					
	
		return $elemsTrouve;
	}
	

	public function restore_first($archivable){
		$nomTable = $archivable->get_archivable_name();
		$dbPath = $this->cheminBase;

		//creation d'un accesseur a la base de donnée
		$ribosome = new Ribosome($dbPath);
		$ribosome->connexion($nomTable);
		
			
		$attributs = $archivable->getAttributs();
		
		
		$elements = $ribosome->donnerTraduction($attributs);			
		
				
	
		foreach ($elements as $element) {
			if($this->correspond($element, $attributs)){
				$archivable = new Archivable($nomTable);
				$archivable->setAttributs($element);
				return $archivable;
			}
		}
	
		return false;
	}
	
	
	/**
	* Renvoi la liste des objet dont les attributs ressemblent au attributs
	* non null de l'objet passé en parametre
	* @return Archivable[] la liste des objet contenu dans la table en fonction des criteres de recherche.
	*/
	public function search($archivable, $chaine_recherche_avancee = NULL){
				
		$nomTable = $archivable->get_archivable_name();
		$dbPath = $this->cheminBase;

		
		//creation d'un accesseur a la base de donnée
		$ribosome = new Ribosome($dbPath);
		$ribosome->connexion($nomTable);
		
		$elemsTrouve = array();		
			
		$attributs = $archivable->getAttributs();
		
		
		$elements = $ribosome->donnerTraduction($attributs);			
		
				
		$nbCles = count($elements);	
		
		$index = 0;
		
		for($i = 0; $i< $nbCles; $i++){			
			//if($this->contient($elements[$i], $attributs)){				
			if($chaine_recherche_avancee == NULL or $this->contient($elements[$i], $attributs)){
	
				$elemsTrouve[$index] = new Archivable($nomTable);
				$elemsTrouve[$index]->setAttributs($elements[$i]);
				$index++;					
			}
		}
					
	
		return $elemsTrouve;
	}	
	
	
	
	/**
	 *  Modifie les attributs des objets archivables $ancien par les attributs de l'objet $nouveau
	 * 
	 * 	La recherche des objets et basé sur les attributs non vides et non null de l'objet $actuel.
	 * 	Seul les attribut non nulls du nouvel objet ecraseron les attributs de l'ancien
	 * 	@return boolean true si une modification à été effectuée
	 *
	 * */
	public function update($ancien, $nouveau){		
		//recuperation du nom de la table
		$nomTable = $nouveau->get_archivable_name();				
		
		$dbPath = $this->cheminBase;

		//creation d'un accesseur a la base de donnée
		$ribosome = new Ribosome($dbPath);
		$ribosome->connexion($nomTable);	
		
		$attributsNew = $nouveau->getAttributs();	
						
		$attributsAncien = $ancien->getAttributs();						
		//recuperation de la traduction de la table sous forme de tableau
		$elements = $ribosome->donnerTraduction($attributsAncien);
					
		
		
		$nbElements = count($elements);
		$nbModif = 0;
		
		for($i = 0; $i< $nbElements; $i++){			
			if($this->correspond($elements[$i], $attributsAncien)){				
				//recuperation et modification des attributs
				$nbModif++;
				$ribosome->supprimer($elements[$i]['id']);
				
				$attributs = $attributsNew;
				foreach($elements[$i] as $cle => $valeur){
					if(!isset($attributs[$cle])){
						$attributs[$cle] = $valeur;
					}
				}
				
				$ribosome->ecrire($attributs);	
				
											
			}
		}				
				
		return $nbModif;
	}

	
	/**
	 * 	Fonction supprimant du systeme de persistance les objet
	 * 	correspondant a l'archivable passé en parametre
	 * 	renvoi true si une suppression à été effectuée
	 */
	public function delete($archivable){	
		
		$nomTable = $archivable->get_archivable_name();					
		
		$dbPath = $this->cheminBase;

		//creation d'un accesseur a la base de donnée
		$ribosome = new Ribosome($dbPath);
		$ribosome->connexion($nomTable);		
		
		//on verifie si on peut utiliser l'index de l'objet
		$id = $archivable->get('id');
		
		$nbSuppr = 0;
		
		//si on cherche l'element par son id
		if (isset($id)){			
			
			$nbSuppr = $ribosome->supprimer($id);
			
		}else{		
			//si on cherche l'element par autre chose			
			//recuperation des attributs sous forme de tableau
			$attributs = $archivable->getAttributs();
					
			
			$elements = $ribosome->donnerTraduction($attributs);				
			$nbElem = count($elements);			
							
			for($i = 0; $i< $nbElem; $i++){			
				if($this->correspond($elements[$i], $attributs)){					
					
					$nbSuppr += $ribosome->supprimer($elements[$i]['id']);
				}
			}						
		}		
		
		return $nbSuppr;
	}
	
	
	
	/**
	 * Permet de trier une liste d'objet archivables en fonction
	 * d'un de ses attributs.
	 * 
	 * @param object[] $liste la liste des objets à trier
	 * @param string $critere le champ sur lequel les objets sont triés
	 * @param boolean $sens le sens du trie. TRUE par defaut.
	 * 					Si $sens = true : ordre alphabetique.
	 * 					Si $sens = false : ordre inverse. 
	 * @return object[] la liste des objets triés.
	 */
	public function sort($liste, $critere, $sens = true){	
	//si sens = true, alpha, sinon tri inverse
		$temp ;	
		$est_modif = true;
		
		while($est_modif){
			$est_modif = false;
			for($i = 1 ; $i < count($liste); $i++){
				
				$comparaison = strcmp ($liste[$i]->get($critere),  $liste[($i-1)]->get($critere));
				if(($sens && $comparaison < 0 ) || (!$sens && $comparaison > 0 )){
					$temp = $liste[$i];
					$liste[$i] = $liste[($i-1)];
					$liste[($i-1)] = $temp ;
					$est_modif = true;
				}
			}
		}
		return $liste; 
	}
	
	public function trierNumCroissant($liste, $critere){	
	
		$temp ;	
		$est_modif = true;
		
		while($est_modif){
			$est_modif = false;
			for($i = 1 ; $i < count($liste); $i++){				
				
				if(intval($liste[$i]->get($critere)) < intval($liste[($i-1)]->get($critere))){
					$temp = $liste[$i];
					$liste[$i] = $liste[($i-1)];
					$liste[($i-1)] = $temp ;
					$est_modif = true;
				}
			}
		}
		return $liste; 
	}
	
	public function trierNumDecroissant($liste, $critere){	
	
		$temp ;	
		$est_modif = true;
		
		while($est_modif){
			$est_modif = false;
			for($i = 1 ; $i < count($liste); $i++){				
				
				if(intval($liste[$i]->get($critere)) > intval($liste[($i-1)]->get($critere))){
					$temp = $liste[$i];
					$liste[$i] = $liste[($i-1)];
					$liste[($i-1)] = $temp ;
					$est_modif = true;
				}
			}
		}
		return $liste; 
	}
	
	private function correspond($elemBase, $elemNouv){		
		$egal = true;		
		foreach($elemNouv as $cle => $valeur){
			if(isset($elemBase[$cle])){
				if($valeur != $elemBase[$cle]){						
					$egal = false;						
				}
			}else{
				$egal = false;
			}			
		}		
		return $egal ;
	}	
	
	private function contient($elemBase, $elemNouv){
		
		
		$egal = false;		
		foreach($elemNouv as $cle => $valeur){
			if(isset($elemBase[$cle])){			
				if(preg_match('/.*'.$valeur.'.*/i', $elemBase[$cle]) ){					
					$egal = true;					
				}				
			}			
		}		
		return $egal ;
	}
	
	

	
	
}
?>